using System.Collections.Generic;
using UnityEngine;

public class SSEManager : MonoBehaviour
{
    private static SSEManager _instance;
    /// <summary>
    /// Singleton instance of the SSEManager.
    /// This ensures that there is only one instance of the SSEManager in the scene.
    /// </summary>
    public static SSEManager Instance
    {
        get
        {
            if (_instance == null)
            {
                var go = new GameObject("SSEManager");
                _instance = go.AddComponent<SSEManager>();
                DontDestroyOnLoad(go);
            }
            return _instance;
        }
    }

    private Dictionary<string, ISSEClient> _clients = new();

    /// <summary>
    /// Gets or creates an SSEClient for the specified URL.
    /// If a client already exists for the URL, it returns the existing client.
    /// If the existing client is of a different type, it logs a warning and returns null.
    /// </summary>
    /// <typeparam name="T">The type of data the client will handle.</typeparam>
    /// <param name="url">The URL for the SSE connection.</param>
    /// <returns>The SSEClient for the specified URL and type, or null if a different type exists.</returns>
    /// <exception cref="SSEClientTypeMismatchException">Thrown when the existing client is of a different type.</exception>
    public SSEClient<T> GetOrCreateClient<T>(string url) where T : class
    {
        if (_clients.TryGetValue(url, out var existingClient))
        {
            if (existingClient is SSEClient<T> typedClient)
                return typedClient;

            throw new SSEClientTypeMismatchException(url, typeof(SSEClient<T>), existingClient.GetType());
        }

        var newClient = new SSEClient<T>(url, this);
        _clients[url] = newClient;
        return newClient;
    }

    /// <summary>
    /// Checks if a client exists for the specified URL.
    /// This method is used to determine if a client has already been created for the given URL.
    /// </summary>
    /// <param name="url">The URL to check for an existing client.</param>
    /// <returns>True if a client exists for the URL, false otherwise.</returns>
    public bool HasClient(string url) => _clients.ContainsKey(url);

    /// <summary>
    /// Starts the SSE client for the specified URL and type.
    /// If the client already exists, it starts the existing client.
    /// If the client does not exist, it creates a new one and starts it.
    /// </summary>
    /// <typeparam name="T">The type of data the client will handle.</typeparam>
    /// <param name="url">The URL for the SSE connection.</param>
    public void StartClient<T>(string url) where T : class
    {
        if (_clients.TryGetValue(url, out var client) && client is SSEClient<T> typedClient)
            typedClient.Start();
    }

    /// <summary>
    /// Stops the SSE client for the specified URL and type.
    /// If the client exists, it stops the existing client.
    /// If the client does not exist, it logs a warning.
    /// </summary>
    /// <typeparam name="T">The type of data the client will handle.</typeparam>
    /// <param name="url">The URL for the SSE connection.</param>
    public void StopClient<T>(string url) where T : class
    {
        if (_clients.TryGetValue(url, out var client) && client is SSEClient<T> typedClient)
            typedClient.Stop();
    }

    /// <summary>
    /// Disposes of the SSE client for the specified URL.
    /// If the client exists, it stops the existing client and removes it from the dictionary.
    /// If the client does not exist, it logs a warning.
    /// </summary>
    /// <param name="url">The URL for the SSE connection.</param>

    public void DisposeClient(string url)
    {
        if (_clients.TryGetValue(url, out var client))
        {
            client.Stop();
            _clients.Remove(url);
        }
    }
}
